function  [dense_cloud_unclustered] = surface_reconstruction(sam_contours, num_gridpoints, ind_smoothness, z_for_slice_1, delta_x_y_z, min_z, flg_view3d)

%% Anirban Chakraborty, Electrical Engineering, University of California, Riverside.


x = [];
y = [];
z = [];
delta_x = delta_x_y_z(1);
delta_y = delta_x_y_z(2);
delta_z = delta_x_y_z(3);

for i=1:length(sam_contours)
    t1 = sam_contours{i}(:,1);
    t2 = sam_contours{i}(:,2);
    x = [x;t1.*delta_x];
    y = [y;t2.*delta_y];
    z = [z;(z_for_slice_1-delta_z*(i-1)).*ones(length(t1),1)];
end;

if length(num_gridpoints) == 1
    num_gridpoints_x = num_gridpoints;
    num_gridpoints_y = num_gridpoints;
else
    num_gridpoints_x = num_gridpoints(1);
    num_gridpoints_y = num_gridpoints(2);
end;

gx=linspace(min(x),max(x),num_gridpoints_x);
gy=linspace(min(y), max(y),num_gridpoints_y);
[zgrid xgrid ygrid] = gridfit(x,y,z,gx,gy, ...
    'smooth',ind_smoothness, ...
    'interp','triangle', ...
    'solver','normal', ...
    'regularizer','gradient', ...
    'extend','never', ...
    'tilesize',inf);

surface_points = [xgrid(:), ygrid(:), zgrid(:)];
surface_points(find(surface_points(:,3)<min_z),:) = [];

dense_cloud_unclustered = sample_points_within_surf(surface_points, delta_z/5, min_z);

if flg_view3d==1
    figure,
    colormap(winter);
    surf(xgrid, ygrid, zgrid);
    camlight right;
    lighting phong;
    shading interp
    view(-90,0)
end;

return;


    function [zgrid,xgrid,ygrid] = gridfit(x,y,z,xnodes,ynodes,varargin)
       
        % https://www.mathworks.com/matlabcentral/fileexchange/8998-surface-fitting-using-gridfit
        % by John D'Errico
        
        % set defaults
        params.smoothness = 1;
        params.interp = 'triangle';
        params.regularizer = 'gradient';
        params.solver = 'backslash';
        params.maxiter = [];
        params.extend = 'warning';
        params.tilesize = inf;
        params.overlap = 0.20;
        params.mask = [];
        params.autoscale = 'on';
        params.xscale = 1;
        params.yscale = 1;
        
        % was the params struct supplied?
        if ~isempty(varargin)
            if isstruct(varargin{1})
                % params is only supplied if its a call from tiled_gridfit
                params = varargin{1};
                if length(varargin)>1
                    % check for any overrides
                    params = parse_pv_pairs(params,varargin{2:end});
                end
            else
                % check for any overrides of the defaults
                params = parse_pv_pairs(params,varargin);
                
            end
        end
        
        % check the parameters for acceptability
        params = check_params(params);
        
        % ensure all of x,y,z,xnodes,ynodes are column vectors,
        % also drop any NaN data
        x=x(:);
        y=y(:);
        z=z(:);
        k = isnan(x) | isnan(y) | isnan(z);
        if any(k)
            x(k)=[];
            y(k)=[];
            z(k)=[];
        end
        xmin = min(x);
        xmax = max(x);
        ymin = min(y);
        ymax = max(y);
        
        % did they supply a scalar for the nodes?
        if length(xnodes)==1
            xnodes = linspace(xmin,xmax,xnodes)';
            xnodes(end) = xmax; % make sure it hits the max
        end
        if length(ynodes)==1
            ynodes = linspace(ymin,ymax,ynodes)';
            ynodes(end) = ymax; % make sure it hits the max
        end
        
        xnodes=xnodes(:);
        ynodes=ynodes(:);
        dx = diff(xnodes);
        dy = diff(ynodes);
        nx = length(xnodes);
        ny = length(ynodes);
        ngrid = nx*ny;
        
        % set the scaling if autoscale was on
        if strcmpi(params.autoscale,'on')
            params.xscale = mean(dx);
            params.yscale = mean(dy);
            params.autoscale = 'off';
        end
        
        % check to see if any tiling is necessary
        if (params.tilesize < max(nx,ny))
            % split it into smaller tiles. compute zgrid and ygrid
            % at the very end if requested
            zgrid = tiled_gridfit(x,y,z,xnodes,ynodes,params);
        else
            % its a single tile.
            
            % mask must be either an empty array, or a boolean
            % aray of the same size as the final grid.
            nmask = size(params.mask);
            if ~isempty(params.mask) && ((nmask(2)~=nx) || (nmask(1)~=ny))
                if ((nmask(2)==ny) || (nmask(1)==nx))
                    error 'Mask array is probably transposed from proper orientation.'
                else
                    error 'Mask array must be the same size as the final grid.'
                end
            end
            if ~isempty(params.mask)
                params.maskflag = 1;
            else
                params.maskflag = 0;
            end
            
            % default for maxiter?
            if isempty(params.maxiter)
                params.maxiter = min(10000,nx*ny);
            end
            
            % check lengths of the data
            n = length(x);
            if (length(y)~=n) || (length(z)~=n)
                error 'Data vectors are incompatible in size.'
            end
            if n<3
                error 'Insufficient data for surface estimation.'
            end
            
            % verify the nodes are distinct
            if any(diff(xnodes)<=0) || any(diff(ynodes)<=0)
                error 'xnodes and ynodes must be monotone increasing'
            end
            
            % do we need to tweak the first or last node in x or y?
            if xmin<xnodes(1)
                switch params.extend
                    case 'always'
                        xnodes(1) = xmin;
                    case 'warning'
                        warning('GRIDFIT:extend',['xnodes(1) was decreased by: ',num2str(xnodes(1)-xmin),', new node = ',num2str(xmin)])
                        xnodes(1) = xmin;
                    case 'never'
                        error(['Some x (',num2str(xmin),') falls below xnodes(1) by: ',num2str(xnodes(1)-xmin)])
                end
            end
            if xmax>xnodes(end)
                switch params.extend
                    case 'always'
                        xnodes(end) = xmax;
                    case 'warning'
                        warning('GRIDFIT:extend',['xnodes(end) was increased by: ',num2str(xmax-xnodes(end)),', new node = ',num2str(xmax)])
                        xnodes(end) = xmax;
                    case 'never'
                        error(['Some x (',num2str(xmax),') falls above xnodes(end) by: ',num2str(xmax-xnodes(end))])
                end
            end
            if ymin<ynodes(1)
                switch params.extend
                    case 'always'
                        ynodes(1) = ymin;
                    case 'warning'
                        warning('GRIDFIT:extend',['ynodes(1) was decreased by: ',num2str(ynodes(1)-ymin),', new node = ',num2str(ymin)])
                        ynodes(1) = ymin;
                    case 'never'
                        error(['Some y (',num2str(ymin),') falls below ynodes(1) by: ',num2str(ynodes(1)-ymin)])
                end
            end
            if ymax>ynodes(end)
                switch params.extend
                    case 'always'
                        ynodes(end) = ymax;
                    case 'warning'
                        warning('GRIDFIT:extend',['ynodes(end) was increased by: ',num2str(ymax-ynodes(end)),', new node = ',num2str(ymax)])
                        ynodes(end) = ymax;
                    case 'never'
                        error(['Some y (',num2str(ymax),') falls above ynodes(end) by: ',num2str(ymax-ynodes(end))])
                end
            end
            
            % determine which cell in the array each point lies in
            [junk,indx] = histc(x,xnodes); %#ok
            [junk,indy] = histc(y,ynodes); %#ok
            % any point falling at the last node is taken to be
            % inside the last cell in x or y.
            k=(indx==nx);
            indx(k)=indx(k)-1;
            k=(indy==ny);
            indy(k)=indy(k)-1;
            ind = indy + ny*(indx-1);
            
            % Do we have a mask to apply?
            if params.maskflag
                % if we do, then we need to ensure that every
                % cell with at least one data point also has at
                % least all of its corners unmasked.
                params.mask(ind) = 1;
                params.mask(ind+1) = 1;
                params.mask(ind+ny) = 1;
                params.mask(ind+ny+1) = 1;
            end
            
            % interpolation equations for each point
            tx = min(1,max(0,(x - xnodes(indx))./dx(indx)));
            ty = min(1,max(0,(y - ynodes(indy))./dy(indy)));
            % Future enhancement: add cubic interpolant
            switch params.interp
                case 'triangle'
                    % linear interpolation inside each triangle
                    k = (tx > ty);
                    L = ones(n,1);
                    L(k) = ny;
                    
                    t1 = min(tx,ty);
                    t2 = max(tx,ty);
                    A = sparse(repmat((1:n)',1,3),[ind,ind+ny+1,ind+L], ...
                        [1-t2,t1,t2-t1],n,ngrid);
                    
                case 'nearest'
                    % nearest neighbor interpolation in a cell
                    k = round(1-ty) + round(1-tx)*ny;
                    A = sparse((1:n)',ind+k,ones(n,1),n,ngrid);
                    
                case 'bilinear'
                    % bilinear interpolation in a cell
                    A = sparse(repmat((1:n)',1,4),[ind,ind+1,ind+ny,ind+ny+1], ...
                        [(1-tx).*(1-ty), (1-tx).*ty, tx.*(1-ty), tx.*ty], ...
                        n,ngrid);
                    
            end
            rhs = z;
            
            % do we have relative smoothing parameters?
            if numel(params.smoothness) == 1
                % it was scalar, so treat both dimensions equally
                smoothparam = params.smoothness;
                xyRelativeStiffness = [1;1];
            else
                % It was a vector, so anisotropy reigns.
                % I've already checked that the vector was of length 2
                smoothparam = sqrt(prod(params.smoothness));
                xyRelativeStiffness = params.smoothness(:)./smoothparam;
            end
            
            % Build regularizer. Add del^4 regularizer one day.
            switch params.regularizer
                case 'springs'
                    % zero "rest length" springs
                    [i,j] = meshgrid(1:nx,1:(ny-1));
                    ind = j(:) + ny*(i(:)-1);
                    m = nx*(ny-1);
                    stiffness = 1./(dy/params.yscale);
                    Areg = sparse(repmat((1:m)',1,2),[ind,ind+1], ...
                        xyRelativeStiffness(2)*stiffness(j(:))*[-1 1], ...
                        m,ngrid);
                    
                    [i,j] = meshgrid(1:(nx-1),1:ny);
                    ind = j(:) + ny*(i(:)-1);
                    m = (nx-1)*ny;
                    stiffness = 1./(dx/params.xscale);
                    Areg = [Areg;sparse(repmat((1:m)',1,2),[ind,ind+ny], ...
                        xyRelativeStiffness(1)*stiffness(i(:))*[-1 1],m,ngrid)];
                    
                    [i,j] = meshgrid(1:(nx-1),1:(ny-1));
                    ind = j(:) + ny*(i(:)-1);
                    m = (nx-1)*(ny-1);
                    stiffness = 1./sqrt((dx(i(:))/params.xscale/xyRelativeStiffness(1)).^2 + ...
                        (dy(j(:))/params.yscale/xyRelativeStiffness(2)).^2);
                    
                    Areg = [Areg;sparse(repmat((1:m)',1,2),[ind,ind+ny+1], ...
                        stiffness*[-1 1],m,ngrid)];
                    
                    Areg = [Areg;sparse(repmat((1:m)',1,2),[ind+1,ind+ny], ...
                        stiffness*[-1 1],m,ngrid)];
                    
                case {'diffusion' 'laplacian'}
                    % thermal diffusion using Laplacian (del^2)
                    [i,j] = meshgrid(1:nx,2:(ny-1));
                    ind = j(:) + ny*(i(:)-1);
                    dy1 = dy(j(:)-1)/params.yscale;
                    dy2 = dy(j(:))/params.yscale;
                    
                    Areg = sparse(repmat(ind,1,3),[ind-1,ind,ind+1], ...
                        xyRelativeStiffness(2)*[-2./(dy1.*(dy1+dy2)), ...
                        2./(dy1.*dy2), -2./(dy2.*(dy1+dy2))],ngrid,ngrid);
                    
                    [i,j] = meshgrid(2:(nx-1),1:ny);
                    ind = j(:) + ny*(i(:)-1);
                    dx1 = dx(i(:)-1)/params.xscale;
                    dx2 = dx(i(:))/params.xscale;
                    
                    Areg = Areg + sparse(repmat(ind,1,3),[ind-ny,ind,ind+ny], ...
                        xyRelativeStiffness(1)*[-2./(dx1.*(dx1+dx2)), ...
                        2./(dx1.*dx2), -2./(dx2.*(dx1+dx2))],ngrid,ngrid);
                    
                case 'gradient'
                    % Subtly different from the Laplacian. A point for future
                    % enhancement is to do it better for the triangle interpolation
                    % case.
                    [i,j] = meshgrid(1:nx,2:(ny-1));
                    ind = j(:) + ny*(i(:)-1);
                    dy1 = dy(j(:)-1)/params.yscale;
                    dy2 = dy(j(:))/params.yscale;
                    
                    Areg = sparse(repmat(ind,1,3),[ind-1,ind,ind+1], ...
                        xyRelativeStiffness(2)*[-2./(dy1.*(dy1+dy2)), ...
                        2./(dy1.*dy2), -2./(dy2.*(dy1+dy2))],ngrid,ngrid);
                    
                    [i,j] = meshgrid(2:(nx-1),1:ny);
                    ind = j(:) + ny*(i(:)-1);
                    dx1 = dx(i(:)-1)/params.xscale;
                    dx2 = dx(i(:))/params.xscale;
                    
                    Areg = [Areg;sparse(repmat(ind,1,3),[ind-ny,ind,ind+ny], ...
                        xyRelativeStiffness(1)*[-2./(dx1.*(dx1+dx2)), ...
                        2./(dx1.*dx2), -2./(dx2.*(dx1+dx2))],ngrid,ngrid)];
                    
            end
            nreg = size(Areg,1);
            
            % Append the regularizer to the interpolation equations,
            % scaling the problem first. Use the 1-norm for speed.
            NA = norm(A,1);
            NR = norm(Areg,1);
            A = [A;Areg*(smoothparam*NA/NR)];
            rhs = [rhs;zeros(nreg,1)];
            % do we have a mask to apply?
            if params.maskflag
                unmasked = find(params.mask);
            end
            % solve the full system, with regularizer attached
            switch params.solver
                case {'\' 'backslash'}
                    if params.maskflag
                        % there is a mask to use
                        zgrid=nan(ny,nx);
                        zgrid(unmasked) = A(:,unmasked)\rhs;
                    else
                        % no mask
                        zgrid = reshape(A\rhs,ny,nx);
                    end
                    
                case 'normal'
                    % The normal equations, solved with \. Can be faster
                    % for huge numbers of data points, but reasonably
                    % sized grids. The regularizer makes A well conditioned
                    % so the normal equations are not a terribly bad thing
                    % here.
                    if params.maskflag
                        % there is a mask to use
                        Aunmasked = A(:,unmasked);
                        zgrid=nan(ny,nx);
                        zgrid(unmasked) = (Aunmasked'*Aunmasked)\(Aunmasked'*rhs);
                    else
                        zgrid = reshape((A'*A)\(A'*rhs),ny,nx);
                    end
                    
                case 'symmlq'
                    % iterative solver - symmlq - requires a symmetric matrix,
                    % so use it to solve the normal equations. No preconditioner.
                    tol = abs(max(z)-min(z))*1.e-13;
                    if params.maskflag
                        % there is a mask to use
                        zgrid=nan(ny,nx);
                        [zgrid(unmasked),flag] = symmlq(A(:,unmasked)'*A(:,unmasked), ...
                            A(:,unmasked)'*rhs,tol,params.maxiter);
                    else
                        [zgrid,flag] = symmlq(A'*A,A'*rhs,tol,params.maxiter);
                        zgrid = reshape(zgrid,ny,nx);
                    end
                    % display a warning if convergence problems
                    switch flag
                        case 0
                            % no problems with convergence
                        case 1
                            % SYMMLQ iterated MAXIT times but did not converge.
                            warning('GRIDFIT:solver',['Symmlq performed ',num2str(params.maxiter), ...
                                ' iterations but did not converge.'])
                        case 3
                            % SYMMLQ stagnated, successive iterates were the same
                            warning('GRIDFIT:solver','Symmlq stagnated without apparent convergence.')
                        otherwise
                            warning('GRIDFIT:solver',['One of the scalar quantities calculated in',...
                                ' symmlq was too small or too large to continue computing.'])
                    end
                    
                case 'lsqr'
                    % iterative solver - lsqr. No preconditioner here.
                    tol = abs(max(z)-min(z))*1.e-13;
                    if params.maskflag
                        % there is a mask to use
                        zgrid=nan(ny,nx);
                        [zgrid(unmasked),flag] = lsqr(A(:,unmasked),rhs,tol,params.maxiter);
                    else
                        [zgrid,flag] = lsqr(A,rhs,tol,params.maxiter);
                        zgrid = reshape(zgrid,ny,nx);
                    end
                    
                    % display a warning if convergence problems
                    switch flag
                        case 0
                            % no problems with convergence
                        case 1
                            % lsqr iterated MAXIT times but did not converge.
                            warning('GRIDFIT:solver',['Lsqr performed ', ...
                                num2str(params.maxiter),' iterations but did not converge.'])
                        case 3
                            % lsqr stagnated, successive iterates were the same
                            warning('GRIDFIT:solver','Lsqr stagnated without apparent convergence.')
                        case 4
                            warning('GRIDFIT:solver',['One of the scalar quantities calculated in',...
                                ' LSQR was too small or too large to continue computing.'])
                    end
                    
            end  % switch params.solver
            
        end  % if params.tilesize...
        
        % only generate xgrid and ygrid if requested.
        if nargout>1
            [xgrid,ygrid]=meshgrid(xnodes,ynodes);
        end
        
        % ============================================
        % End of main function - gridfit
        % ============================================
    end
% ============================================
% subfunction - parse_pv_pairs
% ============================================
    function params=parse_pv_pairs(params,pv_pairs)
        % parse_pv_pairs: parses sets of property value pairs, allows defaults
        % usage: params=parse_pv_pairs(default_params,pv_pairs)
        %
        % arguments: (input)
        %  default_params - structure, with one field for every potential
        %             property/value pair. Each field will contain the default
        %             value for that property. If no default is supplied for a
        %             given property, then that field must be empty.
        %
        %  pv_array - cell array of property/value pairs.
        %             Case is ignored when comparing properties to the list
        %             of field names. Also, any unambiguous shortening of a
        %             field/property name is allowed.
        %
        % arguments: (output)
        %  params   - parameter struct that reflects any updated property/value
        %             pairs in the pv_array.
        %
        % Example usage:
        % First, set default values for the parameters. Assume we
        % have four parameters that we wish to use optionally in
        % the function examplefun.
        %
        %  - 'viscosity', which will have a default value of 1
        %  - 'volume', which will default to 1
        %  - 'pie' - which will have default value 3.141592653589793
        %  - 'description' - a text field, left empty by default
        %
        % The first argument to examplefun is one which will always be
        % supplied.
        %
        %   function examplefun(dummyarg1,varargin)
        %   params.Viscosity = 1;
        %   params.Volume = 1;
        %   params.Pie = 3.141592653589793
        %
        %   params.Description = '';
        %   params=parse_pv_pairs(params,varargin);
        %   params
        %
        % Use examplefun, overriding the defaults for 'pie', 'viscosity'
        % and 'description'. The 'volume' parameter is left at its default.
        %
        %   examplefun(rand(10),'vis',10,'pie',3,'Description','Hello world')
        %
        % params =
        %     Viscosity: 10
        %        Volume: 1
        %           Pie: 3
        %   Description: 'Hello world'
        %
        % Note that capitalization was ignored, and the property 'viscosity'
        % was truncated as supplied. Also note that the order the pairs were
        % supplied was arbitrary.
        
        npv = length(pv_pairs);
        n = npv/2;
        
        if n~=floor(n)
            error 'Property/value pairs must come in PAIRS.'
        end
        if n<=0
            % just return the defaults
            return
        end
        
        if ~isstruct(params)
            error 'No structure for defaults was supplied'
        end
        
        % there was at least one pv pair. process any supplied
        propnames = fieldnames(params);
        lpropnames = lower(propnames);
        for i=1:n
            p_i = lower(pv_pairs{2*i-1});
            v_i = pv_pairs{2*i};
            
            ind = strmatch(p_i,lpropnames,'exact');
            if isempty(ind)
                ind = find(strncmp(p_i,lpropnames,length(p_i)));
                if isempty(ind)
                    error(['No matching property found for: ',pv_pairs{2*i-1}])
                elseif length(ind)>1
                    error(['Ambiguous property name: ',pv_pairs{2*i-1}])
                end
            end
            p_i = propnames{ind};
            
            % override the corresponding default in params
            params = setfield(params,p_i,v_i); %#ok
            
        end
        
    end
% ============================================
% subfunction - check_params
% ============================================
    function params = check_params(params)
        
        % check the parameters for acceptability
        % smoothness == 1 by default
        if isempty(params.smoothness)
            params.smoothness = 1;
        else
            if (numel(params.smoothness)>2) || any(params.smoothness<=0)
                error 'Smoothness must be scalar (or length 2 vector), real, finite, and positive.'
            end
        end
        
        % regularizer  - must be one of 4 options - the second and
        % third are actually synonyms.
        valid = {'springs', 'diffusion', 'laplacian', 'gradient'};
        if isempty(params.regularizer)
            params.regularizer = 'diffusion';
        end
        ind = find(strncmpi(params.regularizer,valid,length(params.regularizer)));
        if (length(ind)==1)
            params.regularizer = valid{ind};
        else
            error(['Invalid regularization method: ',params.regularizer])
        end
        
        % interp must be one of:
        %    'bilinear', 'nearest', or 'triangle'
        % but accept any shortening thereof.
        valid = {'bilinear', 'nearest', 'triangle'};
        if isempty(params.interp)
            params.interp = 'triangle';
        end
        ind = find(strncmpi(params.interp,valid,length(params.interp)));
        if (length(ind)==1)
            params.interp = valid{ind};
        else
            error(['Invalid interpolation method: ',params.interp])
        end
        
        % solver must be one of:
        %    'backslash', '\', 'symmlq', 'lsqr', or 'normal'
        % but accept any shortening thereof.
        valid = {'backslash', '\', 'symmlq', 'lsqr', 'normal'};
        if isempty(params.solver)
            params.solver = '\';
        end
        ind = find(strncmpi(params.solver,valid,length(params.solver)));
        if (length(ind)==1)
            params.solver = valid{ind};
        else
            error(['Invalid solver option: ',params.solver])
        end
        
        % extend must be one of:
        %    'never', 'warning', 'always'
        % but accept any shortening thereof.
        valid = {'never', 'warning', 'always'};
        if isempty(params.extend)
            params.extend = 'warning';
        end
        ind = find(strncmpi(params.extend,valid,length(params.extend)));
        if (length(ind)==1)
            params.extend = valid{ind};
        else
            error(['Invalid extend option: ',params.extend])
        end
        
        % tilesize == inf by default
        if isempty(params.tilesize)
            params.tilesize = inf;
        elseif (length(params.tilesize)>1) || (params.tilesize<3)
            error 'Tilesize must be scalar and > 0.'
        end
        
        % overlap == 0.20 by default
        if isempty(params.overlap)
            params.overlap = 0.20;
        elseif (length(params.overlap)>1) || (params.overlap<0) || (params.overlap>0.5)
            error 'Overlap must be scalar and 0 < overlap < 1.'
        end
    end
% ============================================
% subfunction - tiled_gridfit
% ============================================
    function zgrid=tiled_gridfit(x,y,z,xnodes,ynodes,params)
        % tiled_gridfit: a tiled version of gridfit, continuous across tile boundaries
        % usage: [zgrid,xgrid,ygrid]=tiled_gridfit(x,y,z,xnodes,ynodes,params)
        %
        % Tiled_gridfit is used when the total grid is far too large
        % to model using a single call to gridfit. While gridfit may take
        % only a second or so to build a 100x100 grid, a 2000x2000 grid
        % will probably not run at all due to memory problems.
        %
        % Tiles in the grid with insufficient data (<4 points) will be
        % filled with NaNs. Avoid use of too small tiles, especially
        % if your data has holes in it that may encompass an entire tile.
        %
        % A mask may also be applied, in which case tiled_gridfit will
        % subdivide the mask into tiles. Note that any boolean mask
        % provided is assumed to be the size of the complete grid.
        %
        % Tiled_gridfit may not be fast on huge grids, but it should run
        % as long as you use a reasonable tilesize. 8-)
        
        % Note that we have already verified all parameters in check_params
        
        % Matrix elements in a square tile
        tilesize = params.tilesize;
        % Size of overlap in terms of matrix elements. Overlaps
        % of purely zero cause problems, so force at least two
        % elements to overlap.
        overlap = max(2,floor(tilesize*params.overlap));
        
        % reset the tilesize for each particular tile to be inf, so
        % we will never see a recursive call to tiled_gridfit
        Tparams = params;
        Tparams.tilesize = inf;
        
        nx = length(xnodes);
        ny = length(ynodes);
        zgrid = zeros(ny,nx);
        
        % linear ramp for the bilinear interpolation
        rampfun = inline('(t-t(1))/(t(end)-t(1))','t');
        
        % loop over each tile in the grid
        h = waitbar(0,'Relax and have a cup of JAVA. Its my treat.');
        warncount = 0;
        xtind = 1:min(nx,tilesize);
        while ~isempty(xtind) && (xtind(1)<=nx)
            
            xinterp = ones(1,length(xtind));
            if (xtind(1) ~= 1)
                xinterp(1:overlap) = rampfun(xnodes(xtind(1:overlap)));
            end
            if (xtind(end) ~= nx)
                xinterp((end-overlap+1):end) = 1-rampfun(xnodes(xtind((end-overlap+1):end)));
            end
            
            ytind = 1:min(ny,tilesize);
            while ~isempty(ytind) && (ytind(1)<=ny)
                % update the waitbar
                waitbar((xtind(end)-tilesize)/nx + tilesize*ytind(end)/ny/nx)
                
                yinterp = ones(length(ytind),1);
                if (ytind(1) ~= 1)
                    yinterp(1:overlap) = rampfun(ynodes(ytind(1:overlap)));
                end
                if (ytind(end) ~= ny)
                    yinterp((end-overlap+1):end) = 1-rampfun(ynodes(ytind((end-overlap+1):end)));
                end
                
                % was a mask supplied?
                if ~isempty(params.mask)
                    submask = params.mask(ytind,xtind);
                    Tparams.mask = submask;
                end
                
                % extract data that lies in this grid tile
                k = (x>=xnodes(xtind(1))) & (x<=xnodes(xtind(end))) & ...
                    (y>=ynodes(ytind(1))) & (y<=ynodes(ytind(end)));
                k = find(k);
                
                if length(k)<4
                    if warncount == 0
                        warning('GRIDFIT:tiling','A tile was too underpopulated to model. Filled with NaNs.')
                    end
                    warncount = warncount + 1;
                    
                    % fill this part of the grid with NaNs
                    zgrid(ytind,xtind) = NaN;
                    
                else
                    % build this tile
                    zgtile = gridfit(x(k),y(k),z(k),xnodes(xtind),ynodes(ytind),Tparams);
                    
                    % bilinear interpolation (using an outer product)
                    interp_coef = yinterp*xinterp;
                    
                    % accumulate the tile into the complete grid
                    zgrid(ytind,xtind) = zgrid(ytind,xtind) + zgtile.*interp_coef;
                    
                end
                
                % step to the next tile in y
                if ytind(end)<ny
                    ytind = ytind + tilesize - overlap;
                    % are we within overlap elements of the edge of the grid?
                    if (ytind(end)+max(3,overlap))>=ny
                        % extend this tile to the edge
                        ytind = ytind(1):ny;
                    end
                else
                    ytind = ny+1;
                end
                
            end % while loop over y
            
            % step to the next tile in x
            if xtind(end)<nx
                xtind = xtind + tilesize - overlap;
                % are we within overlap elements of the edge of the grid?
                if (xtind(end)+max(3,overlap))>=nx
                    % extend this tile to the edge
                    xtind = xtind(1):nx;
                end
            else
                xtind = nx+1;
            end
            
        end % while loop over x
        
        % close down the waitbar
        close(h)
        
        if warncount>0
            warning('GRIDFIT:tiling',[num2str(warncount),' tiles were underpopulated & filled with NaNs'])
        end
        
    end
%--------------------------------------------------------------------------
    function pp = splinebase(breaks,n)
        %SPLINEBASE Generate B-spline base PP of order N for breaks BREAKS
        
        breaks = breaks(:);     % Breaks
        breaks0 = breaks';      % Initial breaks
        h = diff(breaks);       % Spacing
        pieces = numel(h);      % Number of pieces
        deg = n - 1;            % Polynomial degree
        
        % Extend breaks periodically
        if deg > 0
            if deg <= pieces
                hcopy = h;
            else
                hcopy = repmat(h,ceil(deg/pieces),1);
            end
            % to the left
            hl = hcopy(end:-1:end-deg+1);
            bl = breaks(1) - cumsum(hl);
            % and to the right
            hr = hcopy(1:deg);
            br = breaks(end) + cumsum(hr);
            % Add breaks
            breaks = [bl(deg:-1:1); breaks; br];
            h = diff(breaks);
            pieces = numel(h);
        end
        
        % Initiate polynomial coefficients
        coefs = zeros(n*pieces,n);
        coefs(1:n:end,1) = 1;
        
        % Expand h
        ii = [1:pieces; ones(deg,pieces)];
        ii = cumsum(ii,1);
        ii = min(ii,pieces);
        H = h(ii(:));
        
        % Recursive generation of B-splines
        for k = 2:n
            % Antiderivatives of splines
            for j = 1:k-1
                coefs(:,j) = coefs(:,j).*H/(k-j);
            end
            Q = sum(coefs,2);
            Q = reshape(Q,n,pieces);
            Q = cumsum(Q,1);
            c0 = [zeros(1,pieces); Q(1:deg,:)];
            coefs(:,k) = c0(:);
            % Normalize antiderivatives by max value
            fmax = repmat(Q(n,:),n,1);
            fmax = fmax(:);
            for j = 1:k
                coefs(:,j) = coefs(:,j)./fmax;
            end
            % Diff of adjacent antiderivatives
            coefs(1:end-deg,1:k) = coefs(1:end-deg,1:k) - coefs(n:end,1:k);
            coefs(1:n:end,k) = 0;
        end
        
        % Scale coefficients
        scale = ones(size(H));
        for k = 1:n-1
            scale = scale./H;
            coefs(:,n-k) = scale.*coefs(:,n-k);
        end
        
        % Reduce number of pieces
        pieces = pieces - 2*deg;
        
        % Sort coefficients by interval number
        ii = [n*(1:pieces); deg*ones(deg,pieces)];
        ii = cumsum(ii,1);
        coefs = coefs(ii(:),:);
        
        % Make piecewise polynomial
        pp = mkpp(breaks0,coefs,n);
    end

%--------------------------------------------------------------------------
    function B = evalcon(base,constr,periodic)
        %EVALCON Evaluate linear constraints
        
        % Unpack structures
        breaks = base.breaks;
        pieces = base.pieces;
        n = base.order;
        xc = constr.xc;
        cc = constr.cc;
        
        % Bin data
        [junk,ibin] = histc(xc,[-inf,breaks(2:end-1),inf]);
        
        % Evaluate constraints
        nx = numel(xc);
        B0 = zeros(n,nx);
        for k = 1:size(cc,1)
            if any(cc(k,:))
                B0 = B0 + repmat(cc(k,:),n,1).*ppval(base,xc);
            end
            % Differentiate base
            coefs = base.coefs(:,1:n-k);
            for j = 1:n-k-1
                coefs(:,j) = (n-k-j+1)*coefs(:,j);
            end
            base.coefs = coefs;
            base.order = n-k;
        end
        
        % Sparse output
        ii = [ibin; ones(n-1,nx)];
        ii = cumsum(ii,1);
        jj = repmat(1:nx,n,1);
        if periodic
            ii = mod(ii-1,pieces) + 1;
            B = sparse(ii,jj,B0,pieces,nx);
        else
            B = sparse(ii,jj,B0,pieces+n-1,nx);
        end
        
    end
%--------------------------------------------------------------------------
    function [Z,u0] = solvecon(B,constr)
        %SOLVECON Find a particular solution u0 and null space Z (Z*B = 0)
        %         for constraint equation u*B = yc.
        
        yc = constr.yc;
        tol = 1000*eps;
        
        % Remove blank rows
        ii = any(B,2);
        B2 = full(B(ii,:));
        
        % Null space of B2
        if isempty(B2)
            Z2 = [];
        else
            % QR decomposition with column permutation
            [Q,R,dummy] = qr(B2); %#ok
            R = abs(R);
            jj = all(R < R(1)*tol, 2);
            Z2 = Q(:,jj)';
        end
        
        % Sizes
        [m,ncon] = size(B);
        m2 = size(B2,1);
        nz = size(Z2,1);
        
        % Sparse null space of B
        Z = sparse(nz+1:nz+m-m2,find(~ii),1,nz+m-m2,m);
        Z(1:nz,ii) = Z2;
        
        % Warning rank deficient
        if nz + ncon > m2
            mess = 'Rank deficient constraints, rank = %d.';
            warning('solvecon:deficient',mess,m2-nz);
        end
        
        % Particular solution
        u0 = zeros(size(yc,1),m);
        if any(yc(:))
            % Non-homogeneous case
            u0(:,ii) = yc/B2;
            % Check solution
            if norm(u0*B - yc,'fro') > norm(yc,'fro')*tol
                mess = 'Inconsistent constraints. No solution within tolerance.';
                error('solvecon:inconsistent',mess)
            end
        end
    end

%--------------------------------------------------------------------------
    function u = lsqsolve(A,y,robust)
        %LSQSOLVE Solve Min norm(u*A-y)
        
        % Avoid sparse-complex limitations
        if issparse(A) && ~isreal(y)
            A = full(A);
        end
        
        % Solution
        u = y/A;
        
        % Robust fitting
        if robust
            [m,n] = size(y);
            for k = 1:3
                % Residual
                r = u*A - y;
                r2 = r.*conj(r);
                r2mean = sum(r2,2)/n;
                r2mean(~r2mean) = 1;
                r2hat = (0.5/m./r2mean)'*r2;
                % Weights
                w = exp(-r2hat);
                spw = spdiags(w',0,n,n);
                % Solve weighted problem
                u = (y*spw)/(A*spw);
            end
        end
        
    end

end

